<?php

namespace App\Http\Controllers\WebAdmin;

use App\Enum\MediaTypeEnum;
use App\Enum\NotificationTypeEnum;
use App\Events\NotifyEvent;
use App\Http\Controllers\Controller;
use App\Http\Requests\CourseStoreRequest;
use App\Http\Requests\CourseUpdateRequest;
use App\Models\Course;
use App\Repositories\CategoryRepository;
use App\Repositories\ContentRepository;
use App\Repositories\CourseRepository;
use App\Repositories\InstructorRepository;
use Illuminate\Support\Facades\Auth;

class CourseController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $course = CourseRepository::query()
            ->when(!$user->hasRole('admin'), function ($query) use ($user) {
                $query->where('instructor_id', $user->instructor?->id);
            })
            ->withTrashed()
            ->latest('id')
            ->get();


        return view('course.index', [
            'courses' => $course,
        ]);
    }

    public function create()
    {
        $user = auth()->user();
        $instructors = InstructorRepository::query()
            ->when(!$user->hasRole('admin'), function ($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->withTrashed()
            ->latest('id')
            ->get();

        return view('course.create', [
            'categories' => CategoryRepository::query()->get(),
            'instructors' => $instructors,
        ]);
    }
    public function show(Course $course)
    {
        $totalEnrollments = $course?->enrollments->count();
        // $totalPrice = $course?->enrollments->sum('course_price');
        $transactions = $course->transactions()->where('is_paid', true)->sum('payment_amount');
        $students = $course->enrollments()->withTrashed()->paginate(15);
        $countClass = $course?->chapters()->count();
        $chapterIds = $course?->chapters()->pluck('id')->toArray();
        $contents = ContentRepository::query()->whereIn('chapter_id', $chapterIds)->get();
        $chapters = $course?->chapters()->get();

        return view('course.overview', [
            'course' => $course,
            "enrollments" => $totalEnrollments,
            "transactions" => $transactions,
            'reviews' => $course->reviews,
            'students' => $students,
            'countClass' => $countClass,
            'durationCount' => $contents->sum('duration'),
            'videoCount' => $contents->where('type', MediaTypeEnum::VIDEO)->count(),
            'imageCount' => $contents->where('type', MediaTypeEnum::IMAGE)->count(),
            'audioCount' => $contents->where('type', MediaTypeEnum::AUDIO)->count(),
            'freeContentCount' => $contents->where('is_free', true)->count(),
            "chapters" => $chapters,
        ]);
    }

    public function store(CourseStoreRequest $request)
    {
        $course = CourseRepository::storeByRequest($request);

        if ($course->is_active) {
            foreach ($course->instructor->courses as $instructorCourse) {
                NotifyEvent::dispatch(NotificationTypeEnum::NewCourseFromInstructor, [
                    'course_id' => $instructorCourse->id,
                ]);
            }
        }

        return to_route('course.index')->with('success', 'Course created');
    }

    public function edit(Course $course)
    {
        return view('course.edit', [
            'course' => $course,
            'categories' => CategoryRepository::query()->get(),
            'instructors' => InstructorRepository::query()->withTrashed()->get(),
        ]);
    }

    public function update(CourseUpdateRequest $request, Course $course)
    {
        CourseRepository::updateByRequest($request, $course);
        if (isset($request->is_active)) {
            foreach ($course->instructor->courses as $instructorCourse) {
                NotifyEvent::dispatch(NotificationTypeEnum::NewCourseFromInstructor, [
                    'course_id' => $instructorCourse->id
                ]);
            }
        }

        return to_route('course.index')->withSuccess('Course updated');
    }

    public function delete(Course $course)
    {
        $course->delete();

        return redirect()->route('course.index')->withSuccess('Course deleted');
    }

    public function restore(int $id)
    {
        CourseRepository::query()->onlyTrashed()->find($id)->restore();

        return redirect()->route('course.index')->withSuccess('Course restored');
    }

    public function freeCourse(Course $course)
    {
        $course->is_free = !$course->is_free;
        $course->updated_at = now();

        $course->save();

        return to_route('course.index')->withSuccess('Course updated');
    }
}
